#coding: utf-8
from django.contrib import admin
from django.db import models
from django.utils.translation import ugettext_lazy as _
from django import forms
from django.db.models import Q
from django.template import Context, Template
from django.conf import settings as django_settings

from peakitcms.conteudos.models import Base, getBaseAdmin, DEFAULT_FIELDSETS, BaseForm
from peakitcms.conteudos.filters import ParentFieldListFilter
from peakitcms.funcionalidades.Mailer import Mailer

from datetime import datetime
from datetime.datetime import timedelta

# Models
def getBaseUtilziador(mymodel, usergroup):
    class BaseUtilizador(models.Model):
        activo = models.BooleanField(_(u'activo'), default=False)
        email = models.EmailField(_(u'email'), unique=True)
        grupos = models.ManyToManyField(mymodel, through=usergroup, null=True, blank=True)

        class Meta:
            abstract = True

        def __unicode__(self):
            return self.email

    return BaseUtilizador

def getBaseGrupo():
    class BaseGrupo(models.Model):
        activo = models.BooleanField(_(u'activo'), default=False)
        titulo = models.CharField(_(u'título'), max_length=255)

        class Meta:
            abstract = True

        def __unicode__(self):
            return self.titulo
    return BaseGrupo

def getBaseUtilizadorGrupo(mymodel_utilizador, mymodel_grupo):
    class BaseUtilizadorGrupo(models.Model):
        utilizador = models.ForeignKey(mymodel_utilizador, verbose_name=_(u'utilizador'), db_index=True)
        grupo = models.ForeignKey(mymodel_grupo, verbose_name=_(u'grupo'), db_index=True)

        class Meta:
            abstract = True
            verbose_name = _(u'grupo')
            verbose_name_plural = _(u'grupos')
            unique_together = (("utilizador", "grupo"),)

        def __unicode__(self):
            return self.grupo.titulo

    return BaseUtilizadorGrupo

TIPOS_NEWSLETTER = (
    (1,_(u'Diária')),
    (2,_(u'Semanal')),
    (3,_(u'Mensal')),
    (4,_(u'Anual')),
    (5,_(u'No Dia')),
)

def getBaseNewsletter(mymodel_utilizador, mymodel_seccoes, mymodel_newsletter_seccao, mymodel_grupo, mymodel_newsletter_grupo, mymodel_queue=None, mymodel_noticias=None, mymodel_pdfs=None, settings_prefix='NEWSLETTER'):
    class BaseNewsletter(models.Model):
        titulo = models.CharField(_(u'título'), max_length=255)
        activa = models.BooleanField(_(u'activa'), default=False)
        assunto = models.CharField(_(u'assunto'), max_length=255)
        corpo = models.TextField(_(u'corpo'))
        tipo = models.IntegerField(_('tipo'), choices=TIPOS_NEWSLETTER, default=1)
        data_inicio = models.DateTimeField(_(u'ínicio'), blank=True, null=True, db_index=True)
        data_enviada = models.DateTimeField(_(u'enviada'), blank=True, null=True, db_index=True)
        enviar_noticias = models.BooleanField(_(u'enviar notícias'), default=False)
        enviar_noticias_x = models.IntegerField(_(u'número de notícias'), blank=True, null=True)
        enviar_pdf_destaque = models.BooleanField(_(u'enviar destaque pdf'), default=False)
        seccoes = models.ManyToManyField(mymodel_seccoes, through=mymodel_newsletter_seccao, null=True, blank=True)
        seccoes.arvore_filter = True
        grupos = models.ManyToManyField(mymodel_grupo, through=mymodel_newsletter_grupo, null=True, blank=True)

        class Meta:
            abstract = True

        def __unicode__(self):
            return self.titulo

        """
        Retorna todos os emails de uma newsletter
        """
        def getallemails(self, n):
            grupos = n.grupos.model.objects.filter(Q(activo=True)).values('id')
            utilizadores = {}
            for g in grupos:
                lusrs = mymodel_utilizador.objects.filter(Q(grupos__id=g['id']), Q(activo=True))
                for u in lusrs:
                    if not u.id in utilizadores:
                        utilizadores[u.id] = u

            return utilizadores

        def parse_newsletter(self, n):
            mail_context = {}
            template = Template(n.corpo)
            mail_context = {}
            meses = {
                     1:u'Janeiro',
                     2:u'Fevereiro',
                     3:u'Março',
                     4:u'Abril',
                     5:u'Maio',
                     6:u'Junho',
                     7:u'Julho',
                     8:u'Agosto',
                     9:u'Setembro',
                     10:u'Outubro',
                     11:u'Novembro',
                     12:u'Dezembro'
                     }
            mail_context['meses'] = meses
            agora = datetime.now()
            hoje = datetime.today()
            mail_context['data'] = {
                                    'dia':agora.day,
                                    'mes':agora.month,
                                    'ano':agora.year,
                                    'mes_extenso': meses[agora.month]
                                    }
            if n.enviar_noticias and not mymodel_noticias is None:
                mail_context['noticias'] = mymodel_noticias.objects.filter(
                                                          Q(activo=True),
                                                          Q(publicada_inicio__lte=agora),
                                                          ).filter(
                                                                   Q(publicada_fim__isnull=True) |
                                                                   Q(publicada_fim__gte=hoje)
                                                                   ).order_by('-publicada_inicio').values('id', 'titulo','capa','resumo','noticia')[:n.enviar_noticias_x]
            if n.enviar_pdf_destaque and not mymodel_pdfs is None:
                c = mymodel_pdfs.objects.filter(
                                               activo=True,
                                               publicada_inicio__lte=agora
                                               ).filter(
                                                        Q(publicada_fim__isnull=True) |
                                                        Q(publicada_fim__gte=hoje)
                                                        ).order_by('-publicada_inicio').values('capa','id')[:1]
                if len(c)==1:
                    mail_context['edicao_pdf'] = c[0]
            return template.render( Context(mail_context) )

        """
        Insere emails de uma newsletter na nossa quee
        """
        def send_newsletter(self, n):
            #self.stdout.write('Iniciar send_newsletter: %s\n' % self.id)
            emails = self.getallemails(n)

            body = self.parse_newsletter(n)

            for e in emails:
                mailer = Mailer(getattr(django_settings, settings_prefix+'_FROM_EMAIL'), emails[e].email, n.assunto, body)
                mailer.dispatcher = getattr(django_settings, settings_prefix+'_DISPATCHER', 'self')
                mailer.queue = mymodel_queue
                mailer.send()
                #self.stdout.write('Inserido: %s\n' % emails[e].email)

            n.data_enviada = datetime.now()
            n.save()
            #self.stdout.write('Fim send_newsletter: %s\n' % self.id)

        """
        Envia emails na nossa quee
        """
        def dispatch_quee(self):
            if mymodel_queue is None:
                raise Exception(u'O parametro mymodel_queue é obrigatório')

            #self.stdout.write('Iniciar dispatch_quee\n')
            if getattr(django_settings, settings_prefix+'_MAX_TO_DISPATCH') is None:
                queue = mymodel_queue.objects.all()
            else:
                queue = mymodel_queue.objects.all()[:getattr(django_settings, settings_prefix+'_MAX_TO_DISPATCH', 500)]

            for e in queue:
                mailer = Mailer(e.de_email, e.para_email, e.assunto, e.corpo)

                if mailer.send() or e.tentativas==3:
                    e.delete()
                else:
                    e.tentativas += 1
                    e.save()
            #self.stdout.write('Fim dispatch_quee\n')

        def ok_newsletter(self, n):
            if n.data_enviada is None: #Não interessa o tipo, se nunca mandou vai mandar
                return True
            dia_enviada = datetime(n.data_enviada.year, n.data_enviada.month, n.data_enviada.day, n.data_inicio.hour, n.data_inicio.minute, n.data_inicio.second)
            hoje = datetime.today()
            if n.tipo==1: #Diária
                z = datetime.now()-dia_enviada
                if z.days>0:
                    return True
            if n.tipo==2: #Semanal
                if dia_enviada<=hoje + timedelta(days=-7):
                    return True
            if n.tipo==3: #Mensal
                if dia_enviada<=hoje + timedelta(days=-30):
                    return True
            if n.tipo==4: #Anual
                if dia_enviada<=hoje + timedelta(days=-365):
                    return True
            return False

        """
        Adiciona todas as newsletter à queue
        """
        def add_quee(self):
            #self.stdout.write('Iniciar add_quee\n')
            agora = datetime.now()
            newsletter = self.__class__.objects.filter(Q(activa=True), Q(data_inicio__lte=agora))
            for n in newsletter:
                if self.ok_newsletter(n):
                    self.send_newsletter(n)
            #self.stdout.write('Fim add_quee\n')

        """
        Trata de decidir o que fazer conforme o que recebe na consola
        """
        def handle(self, *args):
            if 'add_queue' in args:
                #print datetime.now()
                self.add_quee()
            else:
                if 'dispatch_queue' in args:
                    #print datetime.now()
                    self.dispatch_quee()

    return BaseNewsletter

def getBaseNewsletterSeccao(mymodel_newsletter, mymodel_seccao):
    class BaseNewsletterSeccao(models.Model):
        newsletter = models.ForeignKey(mymodel_newsletter, verbose_name=_(u'newsletter'), db_index=True)
        seccao = models.ForeignKey(mymodel_seccao, verbose_name=_(u'secção'), db_index=True)

        class Meta:
            abstract = True
            verbose_name = _(u'secção')
            verbose_name_plural = _(u'secções')
            unique_together = (("newsletter", "seccao"),)

        def __unicode__(self):
            return self.seccao.titulo

    return BaseNewsletterSeccao

def getBaseNewsletterGrupo(mymodel_newsletter, mymodel_grupo):
    class BaseNewsletterGrupo(models.Model):
        newsletter = models.ForeignKey(mymodel_newsletter, verbose_name=_(u'newsletter'), db_index=True)
        grupo = models.ForeignKey(mymodel_grupo, verbose_name=_(u'grupo'), db_index=True)

        class Meta:
            abstract = True
            verbose_name = _(u'grupo')
            verbose_name_plural = _(u'grupos')
            unique_together = (("newsletter", "grupo"),)

        def __unicode__(self):
            return self.grupo.titulo

    return BaseNewsletterGrupo
# Fim Models

# Admin
def getBaseGrupoAdmin():
    class BaseGrupoAdmin(admin.ModelAdmin):
        save_on_top = True
        list_display = ('titulo', 'id')
        list_filter = ('activo',)
        search_fields = ('titulo', 'id')

    return BaseGrupoAdmin

def getBaseGrupoInline(mymodel):
    class BaseGrupoInline(admin.TabularInline):
        extra = 0
        model = mymodel
        raw_id_fields = ('grupo',)
    return BaseGrupoInline

def getBaseUtilizadorAdmin(mymodel):
    class BaseUtilizadorAdmin(admin.ModelAdmin):
        save_on_top = True
        list_display = ('email', 'id')
        list_filter = ('activo', 'grupos',)
        search_fields = ('email', 'id')
        inlines = [getBaseGrupoInline(mymodel),]

    return BaseUtilizadorAdmin

def getBaseSeccaoInline(mymodel):
    class BaseSeccaoInline(admin.TabularInline):
        extra = 0
        model = mymodel
        raw_id_fields = ('seccao',)
    return BaseSeccaoInline

def getBaseNewsletterAdmin(mymodel_grupo, mymodel_seccao):
    class BaseNewsletterAdmin(admin.ModelAdmin):
        save_on_top = True
        list_display = ('titulo', 'activa', 'data_inicio', 'data_enviada', 'tipo', 'id')
        list_filter = ('activa', 'tipo', 'grupos', 'seccoes')
        search_fields = ('titulo', 'id')
        fieldsets = (
                     (None, {
                             'fields':(
                                       'activa',
                                       )
                             }),
                     (u'Agenda', {
                                'fields':(
                                          'tipo',
                                          ('data_inicio', 'data_enviada'),
                                          ),
                                'classes':('collapse',)
                                }),
                     (None, {
                             'fields':(
                                       'titulo',
                                       'assunto',
                                       'corpo',
                                       )
                             }),
                     (u'Extra', {
                                 'fields':(
                                           ('enviar_noticias', 'enviar_noticias_x'),
                                           'enviar_pdf_destaque',
                                           ),
                                 'classes':('collapse',)
                                 }),
                     )
        inlines = [getBaseGrupoInline(mymodel_grupo), getBaseSeccaoInline(mymodel_seccao),]

    return BaseNewsletterAdmin
# Fim Admin
