#coding: utf-8
from django.core.exceptions import ObjectDoesNotExist, ValidationError
from django.shortcuts import render_to_response
from django.template import RequestContext
from elfinder.volume_drivers.base import BaseVolumeDriver
from elfinder.volume_drivers.unzip import unzip
from elfinder import models
import logging
import datetime
import os
import mimetypes
import shutil
from elfinder.connector import ElFinderConnector
try:
    import Image
except:
    from PIL import Image
from collections import namedtuple

logger = logging.getLogger(__name__)


class FsVolumeDriver(BaseVolumeDriver):
    def __init__(self, collection_id,
                 collection_model=models.FileCollection,
                 directory_model=models.Directory,
                 file_model=models.File,
                 *args, **kwargs):

        super(FsVolumeDriver, self).__init__(*args, **kwargs)
        self.collection_model = collection_model
        self.directory_model = directory_model
        self.file_model = file_model

        self.collection = self.collection_model.objects.get(pk=collection_id)
        
    def get_volume_id(self):
        return 'fc%s' % self.collection.id

    def get_info(self, hash):
        return self.get_object(hash).get_info()

    def get_tree(self, target, ancestors=False, siblings=False):
        """ Returns a list of dicts describing children/ancestors/siblings of
            the target directory.

            Siblings of the root node are always excluded, as they refer to
            root directories of other file collections.
        """
        dir = self.get_object(target)
        tree = []
        # Add children to the tree first
        for item in dir.get_children():
            if item.isReal():
                tree.append(item.get_info())
        for item in dir.files.all():
            if item.isReal():
                tree.append(item.get_info())
            
        
        # Add ancestors next, if required
        if ancestors:
            for item in dir.get_ancestors(include_self=True):
                tree.append(item.get_info())
                for ancestor_sibling in item.get_siblings():
                    if ancestor_sibling.parent:
                        tree.append(ancestor_sibling.get_info())

        # Finally add siblings, if required
        if siblings:
            for item in dir.get_siblings():
                if item.parent:
                    tree.append(item.get_info())

        return tree

    def get_object(self, hash):
        """ Returns the object specified by the given hash.

            The hash is in the format "y_xn", where y is the volume_id,
            x is a letter identifying the type of object being requested and
            n is that object's id.

            d:  Directory
            f:  File

            The tree_id of the root node of the currently open FileCollection
            is checked to ensure the target belongs to that tree.
            The client requests the last-remembered dir on init, which breaks
            things if they are now looking at a different FileCollection.

            If the target does not belong to the current tree, return the root
            of the current tree instead.
        """
        if hash == '':
            # No target has been specified so return the root directory.
            return self.directory_model.objects.get(parent=None,
                                                    collection=self.collection)

        try:
            volume_id, object_hash = hash.split('_')
        except ValueError:
            raise Exception('Invalid target hash: %s' % hash)

        try:
            object_id = int(object_hash[1:])
        except ValueError:
            raise Exception('Invalid target hash: %s' % object_hash)

        # Figure which type of object is being requested
        if object_hash[0] == 'f':
            model = self.file_model
        elif object_hash[0] == 'd':
            model = self.directory_model
        else:
            raise Exception('Invalid target hash: %s' % object_hash)

        try:
            object = model.objects.get(pk=object_id,
                                       collection=self.collection.id)
        except ObjectDoesNotExist:
            raise Exception('Could not open target')
        
        if object_hash[0] == 'f':
            pass
        elif object_hash[0] == 'd':
            object.updateFiles()
        return object

    def _create_object(self, name, parent_hash, model):
        """ Helper function to create objects (files/directories).
        """

        parent = self.get_object(parent_hash)

        new_obj = model(name=name,
                        parent=parent,
                        collection=self.collection, last_modification = datetime.datetime.now())
        try:
            new_obj.validate_unique()
            new_obj.save()
        except ValidationError, e:
            #print e
            pass
        except Exception, e:
            #print e
            pass
            #raise Exception("\n".join(e.messages))

        
        return new_obj.get_info()

    def read_file_view(self, request, hash):
        file = self.get_object(hash)
        file.content = open(file.getPath(), 'r').read()
        return render_to_response('read_file.html',
                                  {'file':  file},
                                  RequestContext(request))

    def mkdir(self, name, parent):
        """ Creates a new directory. """
        try:
            p = self.get_object(parent)
            path = p.getPath()
            os.makedirs(path+'/'+name)
	        #print path+'/'+name
            os.chmod(path+'/'+name,0777) #Força a pasta a ter permissões 777


            return self._create_object(name, parent, self.directory_model)
        except:
            return None

    def mkfile(self, name, parent):
        """ Creates a new file. """
        try:
            return self._create_object(name, parent, self.file_model)
        except:
            return None

    def rename(self, name, target):
        try:
            """ Renames a file or directory. """
            object = self.get_object(target)
            os.rename(object.getPath(),object.getDirBefore()+"/"+name)
            object.name = name
            object.save()
            return {'added': [object.get_info()],
                    'removed': [target]}
        except:
            return None

    def list(self, target):
        """ Returns a list of files/directories in the target directory. """
        try:
            list = []
            for object in self.get_tree(target):
                list.append(object['name'])
            return list
        except:
            return None
        

    def paste(self, targets, source, dest, cut):
        try:
            """ Moves/copies target files/directories from source to dest. """
            source_dir = self.get_object(source)
            dest_dir = self.get_object(dest)
            added = []
            removed = []
            for target in targets:

                object = self.get_object(target)
                if cut:
                    os.rename(object.getPath(),dest_dir.getPath()+"/"+object.name)
                else:
                    p = object.getPath()
                    if(os.path.isdir(p)):
                        shutil.copytree(p,dest_dir.getPath())
                    else:
                        shutil.copy(p,dest_dir.getPath())
                object.parent = dest_dir
                if not cut:
                    # This is a copy so the original object should not be changed.
                    # Setting the id to None causes Django to insert a new model
                    # instead of updating the existing one.
                    object.id = None

                # If an object with the same name already exists in the target
                # directory, it should be deleted. This needs to be done for
                # both Files and Directories. Using filter() and iterating
                # over the results is a bit cleaner than using get() and checking
                # if an object was returned, even though most of the time both
                # querysets will be empty.
                dirs = self.directory_model.objects.filter(name=object.name,
                                                       parent=object.parent)
                files = self.file_model.objects.filter(name=object.name,
                                                      parent=object.parent)
                for dir in dirs:
                    removed.append(dir.get_hash())
                    dir.delete()
                for file in files:
                    removed.append(file.get_hash())
                    file.delete()

                object.save()
                added.append(object.get_info())
                if cut:
                    removed.append(object.get_info()['hash'])

            return {'added': added,
                    'removed': removed}
        except:
            return None

    def remove(self, target):
        try:
            """ Delete a File or Directory object. """
            object = self.get_object(target)
            if object.getTipo()=='d':
                os.rmdir(object.getPath())
            else:
                os.remove(object.getPath())
            object.delete()

            return target
        except:
            return None

    
    def _createFile(self,parent,name):
        try:
            caminho = parent.getPath()+"/"+name
            (mode, ino, dev, nlink, uid, gid, size, atime, mtime, ctime) = os.stat(caminho)
            mtime = datetime.datetime.fromtimestamp(mtime)
            mime = mimetypes.guess_type(caminho)

            try:
                new_file  = self.file_model.objects.get(name=name,parent=parent,collection=self.collection)
                new_file.size = size
                new_file.last_modification = mtime
                new_file.mimetype = mime[0]
                #new_file.validate_unique()
            except ValidationError, e:
                logger.exception(e)
                raise Exception("\n".join(e.messages))
            except Exception,e:
                new_file = self.file_model(name=name,
                                       parent=parent,
                                       collection=self.collection,
                                       size=size,
                                       last_modification = mtime,
                                       mimetype = mime[0])

            new_file.save()
            return new_file
        except:
            return None
    
    def upload(self, files, parent):
        try:
            """ For now, this uses a very naive way of storing files - the entire
                file is read in to the File model's content field in one go.

                This should be updated to use read_chunks to add the file one 
                chunk at a time.
            """
            added = []
            parent = self.get_object(parent)
            for upload in files.getlist('upload[]'):
                f = open(parent.getPath()+"/"+upload.name, 'w')
                f.write(upload.read())
                f.close()
                new_file = self._createFile(parent, upload.name)
                added.append(new_file.get_info())
            return {'added': added }
        except:
            return None

    def extract(self,target):
        try:
            obj = self.get_object(target)

            path = obj.getPath()
            mimes = ElFinderConnector().get_init_params()['options']['archivers']['extract'];
            lista = []
            if obj.mimetype in mimes:
                un = unzip()
                lista = un.extract(obj.getPath(), obj.getDirBefore()+'/')
            lista2 = []
            for f in lista:
                new_file = self._createFile(obj.parent, f)
                lista2.append(new_file.get_info())
            #print obj,'::',path,mimes,obj.mimetype
            return lista2#{'added':lista2}
        except:
            return None
    
    def resize(self,target,width,height):
        try:
            width = int(width)
            height = int(height)
            Size = namedtuple('Size', ['width','height'])
            obj = self.get_object(target)
            im1 = Image.open(obj.getPath())
            orig_size = Size(*im1.size)
            if orig_size.width > width*2 or orig_size.height > height*2 :
                im1 = im1.resize(( width*2, height*2 ), Image.NEAREST)
            im1 = im1.resize(( width, height ), Image.ANTIALIAS)
            im1.save(obj.getPath())
            new_file = self._createFile(obj.parent, obj.name)
            return {'changed':[new_file.get_info()]}
        except:
            return None
        
    def crop(self,target,width,height,x,y):
        try: 
            width = int(width)
            height = int(height)
            x = int(x)
            y = int(y)
            obj = self.get_object(target)
            im1 = Image.open(obj.getPath())
            box = (x, y, x+width, y+height)
            im1 = im1.crop(box)
            im1.save(obj.getPath())
            new_file = self._createFile(obj.parent, obj.name)
            return {'changed':[new_file.get_info()]}
        except:
            return None
        
    def rotate(self,target,degree):
        try:
            degree = int(degree)
            obj = self.get_object(target)
            im1 = Image.open(obj.getPath())
            im1 = im1.rotate(degree)
            im1.save(obj.getPath())
            new_file = self._createFile(obj.parent, obj.name)
            return {'changed':[new_file.get_info()]}
        except:
            return None
